function term_structure = convert_swapcurves_termstructure2(swap_curves, swap_maturities, swap_dates, date_nums, T)
% --------------------------------------------------------------------------------------------------
% Same as 'convert_swapcurves_termstructure', but use non-standard maturities for spot rates given
% by T.
% --------------------------------------------------------------------------------------------------
% swap_curves       ... matrix with interest rate swap rates
% swap_maturities   ... maturities corresponding to swap rates
% swap_dates        ... dates corresponding to swap rates
% date_nums         ... date_nums at which term-structure is desired
% T                 ... maturities of interest for each date
% --------------------------------------------------------------------------------------------------
% sample call: convert_swapcurves_termstructure2(swap_curves, swap_maturities, swap_dates, datenum({'11/20/2006'; '11/21/2006'}), [1 5; 0.24 4.9974])
% --------------------------------------------------------------------------------------------------

% Use 1-week swap rate as instantaneous interest rate
swap_curves = [swap_curves(:,1) swap_curves]/100;
swap_maturities = [0; swap_maturities];

term_structure = zeros(0, size(T,2));
available_dates = [];
available_T = [];
was_available = [];
for i=1:length(date_nums)
    % Determine matching swap date
    [trash, pos] = ismember(date_nums(i), datenum(swap_dates));
    if (pos == 0)
        was_available = [was_available; 0];
        continue
    end
    available_dates = [available_dates; date_nums(i)];
    available_T = [available_T; T(i)];
    was_available = [was_available; 1];
    
    % Convert zero rates to forward rates
    zero_rates = swap_curves(pos,:);
    forward_rates = zero_rates(1);
    for j=2:length(zero_rates)
        T_old = swap_maturities(j-1);
        T_new = swap_maturities(j);
        forward_new = ( (1+zero_rates(j))^T_new / (1+zero_rates(j-1))^T_old ) ^ (1/(T_new-T_old)) - 1;
        forward_rates = [forward_rates forward_new];
    end
    
    % Use last forward rate as long-term interest rate
    forward_rates = [forward_rates forward_rates(end)];
    swap_maturities2 = [swap_maturities; 1000];
    
    % Interpolate forward rates using cubic spline
    interpolated_maturities = union(T(i,:), swap_maturities2);
    forward_rates2 = spline(swap_maturities2, forward_rates, interpolated_maturities);
%     plot(swap_maturities, forward_rates2)
%     hold on
%     plot(swap_maturities, forward_rates, 'Color', 'Red')
%     hold off
    
    % Convert forward rates back to zero rates
    zero_rates2 = [];
    for j=1:size(T,2)
        % Determine longest maturities <= T for which zero rate known
        time = T(i,j);
        pos = find(swap_maturities2 <= time, 1, 'last');
        
        % Determine appropriate forward rate for time horizon
        pos2 = find(interpolated_maturities == time, 1, 'last');
        
        % Calculate zero rate for non-standard maturity
        inv_discount = (1+zero_rates(pos)) ^ swap_maturities2(pos) * (1+forward_rates2(pos2)) ^ (time-swap_maturities2(pos));
        zero_rates2(j) = inv_discount ^ (1/time) - 1;
    end
    term_structure = [term_structure; zero_rates2];
%     plot(swap_maturities, zero_rates);
%     hold on
%     plot(T(i,:), zero_rates2, 'Color', 'Red');
%     hold off
end

% Put term-structure into an object
term_structure = struct('zero_rates', term_structure, 'T', available_T, 'dates', {available_dates}, 'was_available', was_available);
